//
//  MeetingSDKMgr.cpp
//  AriesX
//
//  Created by mingfyan on 15/4/24.
//  Copyright (c) 2015 Cisco Systems. All rights reserved.
//

#include "MeetingSDKMgr.h"
#include "csfunified/services/interface/ConfigService.h"
#include "csfunified/services/interface/ConfigValue.h"
#include "csfunified/services/interface/SystemService.h"
#include "csfunified/framework/ServicesDispatcher.h"
#include "csfunified/framework/FunctorTask.h"
#include "jcfcoreutils/FunctionTask.h"

#include "jcfcoreutils/StringUtils.h"
#include "jcfcoreutils/FileUtils.h"
#include "jcfcoreutils/SystemUtils.h"
#include "jcfcoreutils/TimeUtils.h"

#include "csf/logger/CSFLogger.hpp"

#include "SDKAutoUpgrade.h"

#include "../services/impl/MeetingTelemetry.h"

#ifdef _WIN32
#include <Windows.h>
#else
#include <dlfcn.h>
#endif

#include <sstream>
#include <vector>
#include <boost/bind.hpp>

using namespace JCFCoreUtils;

#define CONFIG_MONITOR_URL                       L"MeetingUpgradeServerUrl"
#define CONFIG_MONITOR_INTERVAL                  L"MeetingUpgradeCheckInterval"


#ifdef _WIN32
const std::string MeetingSDKMgr::ENTRY_LIB_NAME = "JabberMeeting.dll";
#else
const std::string MeetingSDKMgr::ENTRY_LIB_NAME = "libjabbermeeting.dylib";
#endif

const std::string MeetingSDKMgr::LOCAL_CONFIG_FOLDER_NAME = "Cisco/MeetingSDK";
const std::string MeetingSDKMgr::LOCAL_CONFIG_FILE_NAME = "meeting_upgrade_info.json";
const std::string MeetingSDKMgr::DEFAULT_SDK_ARCHIVE_NAME = "archive.zip";
const std::string MeetingSDKMgr::INSTALLATION_SDK_SUBFOLDER = "MeetingSDK";

const long MeetingSDKMgr::DEFAULT_URL_SCAN_SECONDS = 60 * 60 * 24;  // default to one day
const long MeetingSDKMgr::LAUNCH_DELAY_SECONDS = 60 * 10;
const long MeetingSDKMgr::AUTOUPGRADE_CHECK_SECONDS = 60;
const long MeetingSDKMgr::DOWNLOAD_WAIT_TIMEOUT_SECONDS = 30 * 60;

static CSFLogger* MeetingSDKMgrLogger = CSFLogger_getLogger("MeetingService-MeetingSDKMgr");

MeetingSDKMgr* MeetingSDKMgr::m_pInstance = NULL;
AutoUpgradeTask* MeetingSDKMgr::m_autoUpgradeTask = NULL;

MeetingSDKMgr::MeetingSDKMgr()
{
    m_localSDKConfigFile = getRemoteSDKFolder() + FileUtils::FileSeparator() + LOCAL_CONFIG_FILE_NAME;
    if (FileUtils::fileExists(m_localSDKConfigFile))
    {
        LocalSDKConfigs::getInstance()->loadFromFile(m_localSDKConfigFile);
    }
    
    m_sdkHandle = NULL;
    m_sdkCheckTrigger = NULL;
    m_sdkCheckerStarted = false;
    
    m_isAutoUpgradeFeatureEnabled = false;
    m_serverUrl = "";
    m_upgradeCheckInterval = 0;
    
    m_pNetUtilTransportDownloader = NULL;
}

std::string MeetingSDKMgr::getRemoteSDKFolder()
{
    std::string localAppDir = FileUtils::getUserLocalAppDir();
    std::string sdkFolder = localAppDir + FileUtils::FileSeparator() + LOCAL_CONFIG_FOLDER_NAME;
    
    return sdkFolder;
}

void MeetingSDKMgr::cleanupRemoteSDKFolder()
{
    std::string remoteFolder = getRemoteSDKFolder();
    if (!FileUtils::isDirectory(remoteFolder))
    {
        return;
    }
    
    std::string remoteVersion = LocalSDKConfigs::getInstance()->getTargetVersion();
    
    std::vector<std::string> subFolders;
    FileUtils::getSubDirectories(remoteFolder, false, subFolders);
    for (std::vector<std::string>::iterator iter = subFolders.begin(); iter != subFolders.end(); iter++)
    {
        if (FileUtils::getFileName(*iter) == remoteVersion)
        {
            continue;
        }
        
        if (FileUtils::isDirectory(*iter))
        {
            FileUtils::removeDirectory(*iter);
        }
    }
}

std::string MeetingSDKMgr::getTargetSDKFolder()
{
    return m_targetSDKFolder;
}

void MeetingSDKMgr::setUnifiedFactory(SMART_PTR_NS::shared_ptr<CSFUnified::UnifiedFactory> unifiedFactory)
{
	m_unifiedFactory = unifiedFactory;
	// instance of the Telemetry
	CMeetingTelemetry MeetingTelemetry(unifiedFactory);
}

void MeetingSDKMgr::getLatestSDKVersion(MeetingSDKVersion& version)
{
    MeetingSDKVersion remoteVersion;
    std::string remoteVersionStr = LocalSDKConfigs::getInstance()->getTargetVersion();
    if (!remoteVersionStr.empty())
    {
        remoteVersion.FromString(remoteVersionStr);
    }
    
    MeetingSDKVersion localVersion;
    getLocalSDKVersion(localVersion);
    
    if (remoteVersion.isLargerThan(localVersion))
    {
        version = remoteVersion;
    }
    else
    {
        version = localVersion;
    }
}

std::string MeetingSDKMgr::getUserDomain()
{
	std::string domain;
	SMART_PTR_NS::shared_ptr<CSFUnified::SystemService> systemService = m_unifiedFactory->getService<CSFUnified::SystemService>();
	if (systemService)
	{
		domain = systemService->getDomainFromUserProfileEmailAddress();
	}
	return domain;
}

void MeetingSDKMgr::getLocalSDKVersion(MeetingSDKVersion& version)
{
    version.major = 5;
    version.middle = 5;
    version.minor = 0;
}

void MeetingSDKMgr::loadJabberConfigs()
{
    // To use config service, the method should be run on main thread
    SMART_PTR_NS::shared_ptr<CSFUnified::ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
    if (pServiceDispatcher)
    {
        bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
        if (bOnDispatcher)
        {
            loadJabberConfigsInternal();
        }
        else
        {
            pServiceDispatcher->enqueueBlock(boost::bind(&MeetingSDKMgr::loadJabberConfigsInternal, this),
                                             "MeetingSDKMgr::loadJabberConfigs()");
        }
    }
}

void MeetingSDKMgr::loadJabberConfigsInternal()
{
    SMART_PTR_NS::shared_ptr<CSFUnified::ConfigService> pConfigService = m_unifiedFactory->getService<CSFUnified::ConfigService>();
    if (!pConfigService)
    {
        return;
    }
    
    std::wstring wValue = L"";
    SMART_PTR_NS::shared_ptr<ConfigValue> configValue = pConfigService->getConfig(CONFIG_MONITOR_URL);
    if (configValue != NULL && configValue->getisDefined())
    {
        wValue = configValue->getValue();
        m_serverUrl = JCFCoreUtils::toString(wValue);
        m_isAutoUpgradeFeatureEnabled = true;
    }
    else
    {
        m_isAutoUpgradeFeatureEnabled = false;
        m_serverUrl = "";
        return;
    }
    
    configValue = pConfigService->getConfig(CONFIG_MONITOR_INTERVAL);
    if (configValue != NULL && configValue->getisDefined())
    {
        wValue = configValue->getValue();
        m_upgradeCheckInterval = JCFCoreUtils::toUnsignedShort(wValue);
    }
    else
    {
        m_upgradeCheckInterval = DEFAULT_URL_SCAN_SECONDS;
    }
}

bool MeetingSDKMgr::loadSDKModule()
{
    loadJabberConfigs();
    
    bool useLocal = false;
    if (!m_isAutoUpgradeFeatureEnabled)
    {
        useLocal = true;
    }
    else
    {
        MeetingSDKVersion sdkVersion;
        getLatestSDKVersion(sdkVersion);
        
        MeetingSDKVersion localVersion;
        getLocalSDKVersion(localVersion);
        
        if (sdkVersion.isEqualTo(localVersion))
        {
            useLocal = true;
            
            long long lastCheckTime = LocalSDKConfigs::getInstance()->getLastCheckTime();
            LocalSDKConfigs::getInstance()->update(lastCheckTime, "", "");
            if (FileUtils::fileExists(m_localSDKConfigFile))
            {
                LocalSDKConfigs::getInstance()->dumpToFile(m_localSDKConfigFile);
            }
        }
    }
    
    if (!useLocal)
    {
        // cleanup remote SDK folder
        cleanupRemoteSDKFolder();
        
        // Try to load from remote firstly
        std::string targetLocation = LocalSDKConfigs::getInstance()->getTargetLocation();
        if (FileUtils::isDirectory(targetLocation) && loadSDKModuleInternal(targetLocation))
        {
            m_targetSDKFolder = targetLocation;
            return true;
        }
        
        // If fail to load remote meeting SDK, remove the corresponding remote SDK folder
        if (!targetLocation.empty()) {
            FileUtils::removeDirectory(targetLocation);
            long long lastCheckTime = LocalSDKConfigs::getInstance()->getLastCheckTime();
            LocalSDKConfigs::getInstance()->update(lastCheckTime, "", "");
            LocalSDKConfigs::getInstance()->dumpToFile(m_localSDKConfigFile);
        }
    }
    
    cleanupRemoteSDKFolder();
    
    // Try to load from local
    std::string installFolder =  FileUtils::getInstallationDirectory();
    if (loadSDKModuleInternal(installFolder))
    {
        m_targetSDKFolder = installFolder;
        return true;
    }
    
    std::string localSDKfolder = installFolder + FileUtils::FileSeparator() + INSTALLATION_SDK_SUBFOLDER;
    if (FileUtils::isDirectory(localSDKfolder) && loadSDKModuleInternal(localSDKfolder))
    {
        m_targetSDKFolder = localSDKfolder;
        return true;
    }

    return false;
}

bool MeetingSDKMgr::loadSDKModuleInternal(std::string folder)
{
    std::string entryLib = folder + FileUtils::FileSeparator() + ENTRY_LIB_NAME;
	if (!FileUtils::fileExists(entryLib))
	{
		return false;
	}

#ifdef _WIN32
    SetDllDirectoryA(folder.c_str());
    m_sdkHandle = LoadLibraryA(entryLib.c_str());
#else
    m_sdkHandle = dlopen(entryLib.c_str(), RTLD_LAZY | RTLD_LOCAL);
#endif
    
    if(m_sdkHandle)
    {
        return true;
    }
    
#ifdef _WIN32
    CSFLogWarnS(MeetingSDKMgrLogger, "Open MeetingSDK lib failed with error code: " << ::GetLastError() << " folder: " << folder);
#else
    CSFLogWarnS(MeetingSDKMgrLogger, "Open MeetingSDK lib failed: " << dlerror() << " folder: " << folder);
#endif
    
    return false;
}

void MeetingSDKMgr::unloadSDKModule()
{
    if (m_sdkHandle)
    {
#ifdef _WIN32
        FreeLibrary(m_sdkHandle);
#else
        dlclose(m_sdkHandle);
#endif
        m_sdkHandle = NULL;
    }
}

NetUtilTransportDownloader* MeetingSDKMgr::getNetUtilTransportDownloader()
{
    if(!m_pNetUtilTransportDownloader && m_unifiedFactory)
    {
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if(isOnDispatcher)
        {
            initNetUtilTransportDownloader();
        }
        else
        {
            csf::TaskPtr task(new JCFCoreUtils::FunctionTask(
                                                             std::bind(&MeetingSDKMgr::initNetUtilTransportDownloader, this),
                                                             "MeetingSDKMgr::initNetUtilTransport"));
            
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(task);
        }
    }
    return m_pNetUtilTransportDownloader;
}

void MeetingSDKMgr::startUpgradeCheck()
{
    CSFLogDebugS(MeetingSDKMgrLogger, "MeetingSDKMgr::startUpgradeCheck()");
    
    if (m_sdkCheckerStarted)
    {
        return;
    }
    
    loadJabberConfigs();
    
    std::string serviceUrl = getServerUrl();
    if (serviceUrl.empty())
    {
        return;
    }
    
    if (!m_autoUpgradeTask)
    {
        m_autoUpgradeTask = new AutoUpgradeTask(this);
    }
    
    m_sdkCheckTrigger = new TaskDelayTrigger(LAUNCH_DELAY_SECONDS, m_autoUpgradeTask, AUTOUPGRADE_CHECK_SECONDS);
    
    m_sdkCheckerStarted = true;
}

void MeetingSDKMgr::stopUpgradeCheck()
{
    if (!m_sdkCheckerStarted) {
        return;
    }
    
	if (m_pNetUtilTransportDownloader)
	{
		m_pNetUtilTransportDownloader->CancelAllRequest();
	}

    if (m_sdkCheckTrigger)
    {
        delete m_sdkCheckTrigger;
        m_sdkCheckTrigger = NULL;
    }
    
    m_sdkCheckerStarted = false;
}

std::string MeetingSDKMgr::getMonitorUrl()
{
    std::stringstream link;
    link << getServerUrl();
    
    std::string osType = "mac";
#ifdef _WIN32
    osType = "windows";
#endif
    
    JabberVersion jabberVersion = SystemUtils::getJabberVersion();
    
    std::stringstream ss;
    ss << jabberVersion.major << "." << jabberVersion.middle << "." << jabberVersion.minor;
    std::string jabberVerString = ss.str();
    
    MeetingSDKVersion sdkVersion;
    getLatestSDKVersion(sdkVersion);
    
    link << "?osType=" << osType << "&jabberVersion=" << jabberVerString << "&sdkVersion=" << sdkVersion.ToString();
    
    // generate a rnd
    long long currentTimeInDays = TimeUtils::getCurrentTimeInSeconds() / (60 * 60 *24);
    link << "&expInterval=" << currentTimeInDays;
    
    SMART_PTR_NS::shared_ptr<CSFUnified::SystemService> systemService = m_unifiedFactory->getService<CSFUnified::SystemService>();
    if (systemService)
    {
        std::string domain = systemService->getDomainFromUserProfileEmailAddress();
        link << "&domain=" << domain;
    }
    
    return link.str();
}

std::string MeetingSDKMgr::getDownloadZipPath()
{
    return getRemoteSDKFolder() + FileUtils::FileSeparator() + DEFAULT_SDK_ARCHIVE_NAME;
}

void MeetingSDKMgr::onQuery(const SDKUpdateQuery& result)
{
    m_lastQueryResult = result;
    LocalSDKConfigs::getInstance()->updateQueryTime(result.currentTimeInSeconds);
    LocalSDKConfigs::getInstance()->dumpToFile(m_localSDKConfigFile);
}

void MeetingSDKMgr::downloadToFileComplete(NetUtilDownloadResultEnum::NetUtilDownloadResult result)
{
    downloadToFileCompleteInternal(result);
    
    m_downloadCompleteEvent.set();
}

void MeetingSDKMgr::downloadToFileCompleteInternal(NetUtilDownloadResultEnum::NetUtilDownloadResult result)
{
    std::string zipfile = getDownloadZipPath();

	MeetingSDKVersion localSDKVersion;
	std::string userDomain;
	getLatestSDKVersion(localSDKVersion);
	userDomain = getUserDomain();
    
    if (result != NetUtilDownloadResultEnum::OK)
    {
        CSFLogErrorS(MeetingSDKMgrLogger, "Download Meeting SDK package failed." );
        FileUtils::removeFile(zipfile);

		CMeetingTelemetry telemetry;
		telemetry.addMember("host_domain", userDomain);
		telemetry.addMember("current_version", localSDKVersion.ToString());
		telemetry.addMember("upgrade_version", m_lastQueryResult.version);
		telemetry.addMember("need_upgrade", "yes");
		telemetry.addMember("upgrade_result", "failure");
		telemetry.addMember("failure_reason", "Download Meeting SDK package failed");
		telemetry.addMember("failure_phase", "download meetingsdk phase");
		telemetry.upload("meetingupgrade");

        return;
    }
    
    // check checksum of file
    std::string cksum = FileCheckerUtils::calc_file_sha256(zipfile);
    if (cksum != m_lastQueryResult.hashKey) {
        CSFLogErrorS(MeetingSDKMgrLogger, "Downloaded Meeting SDK package SHA256 Checksum doesn't match." );
        FileUtils::removeFile(zipfile);

		CMeetingTelemetry telemetry;
		telemetry.addMember("host_domain", userDomain);
		telemetry.addMember("current_version", localSDKVersion.ToString());
		telemetry.addMember("upgrade_version", m_lastQueryResult.version);
		telemetry.addMember("need_upgrade", "yes");
		telemetry.addMember("upgrade_result", "failure");
		telemetry.addMember("failure_reason", "Downloaded Meeting SDK package SHA256 Checksum doesn't match.");
		telemetry.addMember("failure_phase", "download meetingsdk phase");
		telemetry.upload("meetingupgrade");

        return;
    }
    
    // uncompress file
    std::string version = m_lastQueryResult.version;
    std::string targetUncompressFolder = getRemoteSDKFolder() + FileUtils::FileSeparator() + version;
    if (!FileUtils::isDirectory(targetUncompressFolder))
    {
        FileUtils::createDirectory(targetUncompressFolder);
    }
    
    UnzipUtils unzipUtils;
    bool ret = unzipUtils.UnZipWithoutPassword(zipfile, targetUncompressFolder);
    FileUtils::removeFile(zipfile);
    if (!ret)
    {
        CSFLogErrorS(MeetingSDKMgrLogger, "Failed to uncompress meeting SDK zip file " <<  zipfile);

		CMeetingTelemetry telemetry;
		telemetry.addMember("host_domain", userDomain);
		telemetry.addMember("current_version", localSDKVersion.ToString());
		telemetry.addMember("upgrade_version", m_lastQueryResult.version);
		telemetry.addMember("need_upgrade", "yes");
		telemetry.addMember("upgrade_result", "failure");
		telemetry.addMember("failure_reason", "Failed to uncompress meeting SDK zip file.");
		telemetry.addMember("failure_phase", "install meetingsdk phase");
		telemetry.upload("meetingupgrade");

        return;
    }
    
    // check files signature in uncompressed folder
    ret = FileCheckerUtils::checkSignatureForFolderFiles(targetUncompressFolder);
    if (!ret)
    {
        CSFLogErrorS(MeetingSDKMgrLogger, "Check signature failed for files in uncompressed folder " <<  targetUncompressFolder);
        FileUtils::removeDirectory(targetUncompressFolder);

		CMeetingTelemetry telemetry;
		telemetry.addMember("host_domain", userDomain);
		telemetry.addMember("current_version", localSDKVersion.ToString());
		telemetry.addMember("upgrade_version", m_lastQueryResult.version);
		telemetry.addMember("need_upgrade", "yes");
		telemetry.addMember("upgrade_result", "failure");
		telemetry.addMember("failure_reason", "Check signature failed.");
		telemetry.addMember("failure_phase", "install meetingsdk phase");
		telemetry.upload("meetingupgrade");

        return;
    }
    
    // rewrite SDK local config file
    long long currentTimeInSeconds = m_lastQueryResult.currentTimeInSeconds;

	CMeetingTelemetry telemetry;
	telemetry.addMember("host_domain", userDomain);
	telemetry.addMember("current_version", localSDKVersion.ToString());
	telemetry.addMember("upgrade_version", m_lastQueryResult.version);
	telemetry.addMember("need_upgrade", "yes");
	telemetry.addMember("upgrade_result", "success");
	telemetry.upload("meetingupgrade");

    LocalSDKConfigs::getInstance()->update(currentTimeInSeconds, version, targetUncompressFolder);
    LocalSDKConfigs::getInstance()->dumpToFile(m_localSDKConfigFile);
}
